<?php

namespace BitApps\PiPro\src\Integrations\Discord;

use BitApps\Pi\Deps\BitApps\WPKit\Helpers\JSON;
use BitApps\Pi\Helpers\Utility;

if (!\defined('ABSPATH')) {
    exit;
}

final class DiscordMessageService
{
    private $http;

    private $baseUrl;

    public function __construct($httpClient, $baseUrl)
    {
        $this->http = $httpClient;
        $this->baseUrl = $baseUrl;
    }

    public function sendMessage($channelId, $data)
    {
        $response = $this->http->request(
            $this->baseUrl . "/channels/{$channelId}/messages",
            'POST',
            JSON::encode($data),
            []
        );

        $responseCode = $this->http->getResponseCode();
        $success = $responseCode >= 200 && $responseCode < 300;

        return Utility::formatResponseData(
            $responseCode,
            $data,
            $response,
            $success ? 'Message sent successfully' : $response->message
        );
    }

    public function sendDirectMessage($userId, $data)
    {
        $response = $this->http->request(
            $this->baseUrl . '/users/@me/channels',
            'POST',
            JSON::encode(
                [
                    'recipient_id' => $userId,
                    'content' => $data['content']
                ]
            ),
            []
        );

        return $this->sendMessage($response->id, $data);
    }

    public function deleteMessage($channelId, $messageId)
    {
        $response = $this->http->request(
            $this->baseUrl . "/channels/{$channelId}/messages/{$messageId}",
            'DELETE',
            []
        );

        $payload = ['channel_id' => $channelId, 'message_id' => $messageId];

        $responseCode = $this->http->getResponseCode();
        $success = $responseCode >= 200 && $responseCode < 300;

        return Utility::formatResponseData(
            $responseCode,
            $payload,
            $response,
            $success ? 'Message deleted successfully' : $response->message
        );
    }

    public function getMessage($channelId, $messageId)
    {
        $response = $this->http->request(
            $this->baseUrl . "/channels/{$channelId}/messages/{$messageId}",
            'GET',
            []
        );

        $payload = ['channel_id' => $channelId, 'message_id' => $messageId];

        $responseCode = $this->http->getResponseCode();
        $success = $responseCode >= 200 && $responseCode < 300;

        return Utility::formatResponseData(
            $responseCode,
            $payload,
            $response,
            $success ? 'Message retrieved successfully' : $response->message
        );
    }

    public function getMessages($channelId, $limit = 50)
    {
        $limit = max(1, min(100, (int) $limit));
        $response = $this->http->request(
            $this->baseUrl . "/channels/{$channelId}/messages?limit={$limit}",
            'GET',
            []
        );

        $payload = ['channel_id' => $channelId, 'limit' => $limit];

        $responseCode = $this->http->getResponseCode();
        $success = $responseCode >= 200 && $responseCode < 300;

        return Utility::formatResponseData(
            $responseCode,
            $payload,
            $response,
            $success ? 'Messages retrieved successfully' : $response->message
        );
    }

    public function reactWithEmoji($channelId, $messageId, $emoji)
    {
        $encodedEmoji = urlencode($emoji);

        $response = $this->http->request(
            $this->baseUrl . "/channels/{$channelId}/messages/{$messageId}/reactions/{$encodedEmoji}/@me",
            'PUT',
            []
        );
        $payload = ['channel_id' => $channelId, 'message_id' => $messageId, 'emoji' => $emoji];

        $responseCode = $this->http->getResponseCode();
        $success = $responseCode >= 200 && $responseCode < 300;

        return Utility::formatResponseData(
            $responseCode,
            $payload,
            $success ? true : null,
            $success ? 'Reaction added successfully' : $response->message
        );
    }
}
