<?php

namespace BitApps\PiPro\src\Integrations\Dropbox;

// Prevent direct script access
if (!\defined('ABSPATH')) {
    exit;
}

use BitApps\Pi\Helpers\Utility;
use BitApps\Pi\src\Authorization\AuthorizationFactory;
use BitApps\Pi\src\Authorization\AuthorizationType;
use BitApps\Pi\src\Flow\NodeInfoProvider;
use BitApps\Pi\src\Interfaces\ActionInterface;
use BitApps\PiPro\Deps\BitApps\WPKit\Http\Client\HttpClient;
use InvalidArgumentException;

class DropboxAction implements ActionInterface
{
    private NodeInfoProvider $nodeInfoProvider;

    public function __construct(NodeInfoProvider $nodeInfoProvider)
    {
        $this->nodeInfoProvider = $nodeInfoProvider;
    }

    public function execute(): array
    {
        $executedNodeAction = $this->executeDropboxAction();

        return $executedNodeAction;
    }

    private function executeDropboxAction(): array
    {
        $machineSlug = $this->nodeInfoProvider->getMachineSlug();

        $connectionId = $this->nodeInfoProvider->getFieldMapConfigs('connection-id.value');

        if (!$connectionId) {
            return Utility::formatResponseData(
                400,
                [],
                ['error' => 'Connection not found'],
                __('Connection not found', 'bit-pi')
            );
        }

        $accessToken = AuthorizationFactory::getAuthorizationHandler(
            AuthorizationType::OAUTH2,
            $connectionId
        )->getAccessToken();

        $httpClient = new HttpClient();
        $dropboxService = new DropboxService($httpClient, $accessToken);

        return $this->executeAction($machineSlug, $dropboxService);
    }

    private function executeAction(string $machineSlug, DropboxService $dropboxService): array
    {
        switch ($machineSlug) {
            case 'uploadFile':
                return $dropboxService->uploadFile($this->nodeInfoProvider);

            case 'downloadFile':
                return $dropboxService->downloadFile($this->nodeInfoProvider);

            case 'deleteFile':
                return $dropboxService->deleteFile($this->nodeInfoProvider);

            case 'moveFile':
                return $dropboxService->moveFile($this->nodeInfoProvider);

            case 'copyFile':
                return $dropboxService->copyFile($this->nodeInfoProvider);

            case 'createFolder':
                return $dropboxService->createFolder($this->nodeInfoProvider);

            case 'deleteFolder':
                return $dropboxService->deleteFolder($this->nodeInfoProvider);

            case 'moveFolder':
                return $dropboxService->moveFolder($this->nodeInfoProvider);

            case 'copyFolder':
                return $dropboxService->copyFolder($this->nodeInfoProvider);

            case 'listFolder':
                return $dropboxService->listFolder($this->nodeInfoProvider);

            case 'searchFiles':
                return $dropboxService->searchFiles($this->nodeInfoProvider);

            default:
                throw new InvalidArgumentException("Unknown action: {$machineSlug}");
        }
    }
}
