<?php

namespace BitApps\PiPro\src\Integrations\FluentCart;

// Prevent direct script access
if (!\defined('ABSPATH')) {
    exit;
}

use BitApps\Pi\Helpers\Utility;
use BitApps\Pi\Services\FlowService;
use BitApps\PiPro\src\Integrations\IntegrationHelper;

final class FluentCartTrigger
{
    public static function productPurchased(array $data): void
    {
        if (empty($data['cart']) || empty($data['order'])) {
            return;
        }

        $data['cart'] = FluentCartHelper::prepareData($data['cart']);
        $data['order'] = FluentCartHelper::prepareData($data['order']);

        self::execute(__FUNCTION__, $data);
    }

    public static function productUpdated(array $data): void
    {
        self::executeProductUpdatedTrigger(__FUNCTION__, $data);
    }

    public static function productStockUpdated(array $data): void
    {
        self::executeProductUpdatedTrigger(__FUNCTION__, $data);
    }

    public static function couponCreated(array $data): void
    {
        self::executeCouponTrigger(__FUNCTION__, $data);
    }

    public static function couponUpdated(array $data): void
    {
        self::executeCouponTrigger(__FUNCTION__, $data);
    }

    public static function cartItemAdded(array $data): void
    {
        self::executeCartTriggers(__FUNCTION__, $data);
    }

    public static function cartItemRemoved(array $data): void
    {
        self::executeCartTriggers(__FUNCTION__, $data);
    }

    public static function cartItemsUpdated(array $data): void
    {
        self::executeCartTriggers(__FUNCTION__, $data);
    }

    public static function cartUpdated(array $data): void
    {
        self::executeCartTriggers(__FUNCTION__, $data);
    }

    public static function cartCompleted(array $data): void
    {
        if (empty($data['cart']) || empty($data['order'])) {
            return;
        }

        $data['cart'] = FluentCartHelper::prepareData($data['cart']);
        $data['order'] = FluentCartHelper::prepareData($data['order']);

        self::execute(__FUNCTION__, $data);
    }

    public static function orderReceiptRendered(array $data): void
    {
        if (empty($data['order'])) {
            return;
        }

        $data['order'] = FluentCartHelper::prepareData($data['order']);
        $data['order_operation'] = FluentCartHelper::prepareData($data['order_operation'] ?? []);

        self::execute(__FUNCTION__, $data);
    }

    public static function orderCreated(array $data): void
    {
        self::executeOrderPaymentTrigger(__FUNCTION__, $data);
    }

    public static function orderUpdated(array $data): void
    {
        if (empty($data['order'])) {
            return;
        }

        $data['order'] = FluentCartHelper::prepareData($data['order']);
        $data['old_order'] = FluentCartHelper::prepareData($data['old_order'] ?? []);

        self::execute(__FUNCTION__, $data);
    }

    public static function orderDeleted(array $data): void
    {
        if (empty($data['order'])) {
            return;
        }

        $data['order'] = FluentCartHelper::prepareData($data['order']);
        $data['customer'] = FluentCartHelper::prepareData($data['customer'] ?? []);

        self::execute(__FUNCTION__, $data);
    }

    public static function orderRefunded(array $data): void
    {
        self::executeOrderPaymentTrigger(__FUNCTION__, $data);
    }

    public static function orderPartiallyRefunded(array $data): void
    {
        self::executeOrderPaymentTrigger(__FUNCTION__, $data);
    }

    public static function orderFullyRefunded(array $data): void
    {
        self::executeOrderPaymentTrigger(__FUNCTION__, $data);
    }

    public static function orderCancelled(array $data): void
    {
        self::executeOrderStatusTrigger(__FUNCTION__, $data);
    }

    public static function orderStatusChanged(array $data): void
    {
        self::executeOrderStatusTrigger(__FUNCTION__, $data);
    }

    public static function orderStatusUpdatedToProcessing(array $data): void
    {
        self::executeOrderStatusTrigger(__FUNCTION__, $data);
    }

    public static function orderStatusUpdatedToCompleted(array $data): void
    {
        self::executeOrderStatusTrigger(__FUNCTION__, $data);
    }

    public static function orderStatusUpdatedToOnHold(array $data): void
    {
        self::executeOrderStatusTrigger(__FUNCTION__, $data);
    }

    public static function orderShippingStatusChanged(array $data): void
    {
        self::executeOrderStatusTrigger(__FUNCTION__, $data);
    }

    public static function orderShipped(array $data): void
    {
        self::executeOrderStatusTrigger(__FUNCTION__, $data);
    }

    public static function orderUnshipped(array $data): void
    {
        self::executeOrderStatusTrigger(__FUNCTION__, $data);
    }

    public static function orderUnshippable(array $data): void
    {
        self::executeOrderStatusTrigger(__FUNCTION__, $data);
    }

    public static function orderDelivered(array $data): void
    {
        self::executeOrderStatusTrigger(__FUNCTION__, $data);
    }

    public static function orderMarkedAsPaid(array $data): void
    {
        self::executeOrderPaymentTrigger(__FUNCTION__, $data);
    }

    public static function orderCustomerChanged(array $data): void
    {
        if (empty($data['order']) || empty($data['old_customer']) || empty($data['new_customer'])) {
            return;
        }

        $data['order'] = FluentCartHelper::prepareData($data['order']);
        $data['old_customer'] = FluentCartHelper::prepareData($data['old_customer']);
        $data['new_customer'] = FluentCartHelper::prepareData($data['new_customer']);

        self::execute(__FUNCTION__, $data);
    }

    public static function customerCreated(int $userId): void
    {
        self::execute(__FUNCTION__, Utility::getUserInfo($userId));
    }

    /**
     * Execute cart related triggers.
     *
     * @param string $machineSlug
     * @param array  $data
     */
    private static function executeCartTriggers($machineSlug, $data)
    {
        if (empty($data['cart'])) {
            return;
        }

        $data['cart'] = FluentCartHelper::prepareData($data['cart']);

        self::execute($machineSlug, $data);
    }

    /**
     * Execute product updated triggers.
     *
     * @param string $machineSlug
     * @param array  $data
     */
    private static function executeProductUpdatedTrigger($machineSlug, $data)
    {
        if (empty($data['product'])) {
            return;
        }

        $data['product'] = FluentCartHelper::prepareData($data['product']);

        self::execute($machineSlug, $data);
    }

    /**
     * Execute coupon related triggers.
     *
     * @param string $machineSlug
     * @param array  $data
     */
    private static function executeCouponTrigger($machineSlug, $data)
    {
        if (empty($data['coupon'])) {
            return;
        }

        $data['coupon'] = FluentCartHelper::prepareData($data['coupon']);

        self::execute($machineSlug, $data);
    }

    /**
     * Execute order status related triggers.
     *
     * @param string $machineSlug
     * @param array  $data
     */
    private static function executeOrderStatusTrigger($machineSlug, $data)
    {
        if (empty($data['order'])) {
            return;
        }

        $data['order'] = FluentCartHelper::prepareData($data['order']);

        self::execute($machineSlug, $data);
    }

    /**
     * Execute order payment related triggers.
     *
     * @param string $machineSlug
     * @param array  $data
     */
    private static function executeOrderPaymentTrigger($machineSlug, $data)
    {
        if (empty($data['order']) || empty($data['customer']) || empty($data['transaction'])) {
            return;
        }

        $data['order'] = FluentCartHelper::prepareData($data['order']);
        $data['customer'] = FluentCartHelper::prepareData($data['customer']);
        $data['transaction'] = FluentCartHelper::prepareData($data['transaction']);

        self::execute($machineSlug, $data);
    }

    /**
     * Execute the trigger.
     *
     * @param string $machineSlug
     * @param array  $data
     */
    private static function execute($machineSlug, $data)
    {
        $flows = FlowService::exists('FluentCart', $machineSlug);

        if (!$flows) {
            return;
        }

        IntegrationHelper::handleFlowForForm($flows, $data);
    }
}
