<?php

namespace BitApps\PiPro\src\Integrations\FluentCart\Services;

use BitApps\Pi\src\Flow\NodeInfoProvider;
use BitApps\PiPro\src\Integrations\FluentCart\FluentCartHelper;
use FluentCart\App\Models\Coupon;

if (!\defined('ABSPATH')) {
    exit;
}

final class CouponService
{
    private $nodeInfoProvider;

    public function __construct(NodeInfoProvider $nodeInfoProvider)
    {
        $this->nodeInfoProvider = $nodeInfoProvider;
    }

    public function getAllCoupon(): array
    {
        if ($validationResponse = FluentCartHelper::validateClassExistence('coupon')) {
            return $validationResponse;
        }

        $coupons = Coupon::all();
        $data = ['coupons' => FluentCartHelper::prepareData($coupons)];

        return [
            'response'    => $data,
            'payload'     => [],
            'status_code' => 200
        ];
    }

    public function getCouponById(): array
    {
        $coupon = $this->validateAndGetCoupon();

        if (!$coupon instanceof Coupon) {
            return $coupon;
        }

        return [
            'response'    => FluentCartHelper::prepareData($coupon),
            'payload'     => [],
            'status_code' => 200
        ];
    }

    public function createCoupon(): array
    {
        if ($validationError = FluentCartHelper::validateClassExistence('coupon')) {
            return $validationError;
        }

        $fieldMappings = $this->nodeInfoProvider->getFieldMapData();
        $rules = [
            'title'  => ['required', 'string'],
            'code'   => ['required', 'string'],
            'type'   => ['required', 'string'],
            'amount' => ['required', 'integer'],
        ];

        if ($validationError = FluentCartHelper::validateFieldMap($fieldMappings, $rules)) {
            return $validationError;
        }

        $fieldMappings['status'] = 'active';

        $coupon = Coupon::create($fieldMappings);

        if (!$coupon) {
            return [
                'response'    => __('Failed to create the coupon.', 'bit-pi'),
                'payload'     => $fieldMappings,
                'status_code' => 500
            ];
        }

        return [
            'response'    => FluentCartHelper::prepareData($coupon),
            'payload'     => $fieldMappings,
            'status_code' => 200
        ];
    }

    public function deleteCoupon(): array
    {
        $coupon = $this->validateAndGetCoupon();

        if (!$coupon instanceof Coupon) {
            return $coupon;
        }

        if (!$coupon->delete()) {
            return [
                'response'    => __('Failed to delete the coupon.', 'bit-pi'),
                'payload'     => [],
                'status_code' => 500
            ];
        }

        return [
            'response'    => __('Coupon deleted successfully.', 'bit-pi'),
            'payload'     => [],
            'status_code' => 200
        ];
    }

    /**
     * Validate and get Coupon.
     */
    private function validateAndGetCoupon(): array|Coupon
    {
        if ($validationError = FluentCartHelper::validateClassExistence('coupon')) {
            return $validationError;
        }

        $fieldMappings = $this->nodeInfoProvider->getFieldMapData();
        $rules = ['id' => ['required', 'integer']];

        if ($validationError = FluentCartHelper::validateFieldMap($fieldMappings, $rules)) {
            return $validationError;
        }

        $coupon = Coupon::find($fieldMappings['id']);

        if (!$coupon) {
            return [
                'response'    => __('Coupon not found.', 'bit-pi'),
                'payload'     => [],
                'status_code' => 404
            ];
        }

        return $coupon;
    }
}
