<?php

namespace BitApps\PiPro\src\Integrations\GoHighLevel;

use BitApps\PiPro\Deps\BitApps\WPKit\Helpers\JSON;
use BitApps\PiPro\Deps\BitApps\WPKit\Http\Client\HttpClient;

if (!\defined('ABSPATH')) {
    exit;
}

class GoHighLevelV2Service
{
    private HttpClient $http;

    private string $baseUrl;

    /**
     * GoHighLevelV2Service constructor.
     *
     * @param HttpClient $httpClient
     */
    public function __construct($httpClient)
    {
        $this->http = $httpClient;
        $this->baseUrl = 'https://services.leadconnectorhq.com/';
    }

    /**
     * Add a contact using OAuth2.
     *
     * @param array $fieldMapData
     *
     * @return array
     */
    public function addContact($fieldMapData)
    {
        $endPoint = $this->baseUrl . 'contacts/';

        $response = $this->http->request(
            $endPoint,
            'POST',
            JSON::encode($fieldMapData)
        );

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Add a contact using OAuth2.
     *
     * @param array $fieldMapData
     * @param mixed $locationId
     *
     * @return array
     */
    public function searchContacts($fieldMapData, $locationId)
    {
        $endPoint = $this->baseUrl . 'contacts/';

        $queryParams = [
            'query'      => $fieldMapData['query'] ?? null,
            'locationId' => $locationId ?? null,
            'limit'      => 100
        ];

        $queryParams = array_filter($queryParams);
        $queryString = http_build_query($queryParams);
        $endPoint = $endPoint . '?' . $queryString;
        $response = $this->http->request($endPoint, 'GET', []);

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Add a contact using OAuth2.
     *
     * @param array $fieldMapData
     * @param mixed $contactId
     *
     * @return array
     */
    public function addNoteToContact($contactId, $fieldMapData)
    {
        $endPoint = $this->baseUrl . 'contacts/' . $contactId . '/notes';
        $response = $this->http->request(
            $endPoint,
            'POST',
            JSON::encode($fieldMapData)
        );

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Add a contact to a campaign using OAuth2.
     *
     * @param string $contactId
     * @param string $campaignId
     *
     * @return array
     */
    public function addContactToCampaign($contactId, $campaignId)
    {
        $endPoint = $this->baseUrl . 'contacts/' . $contactId . '/campaigns/' . $campaignId;

        $response = $this->http->request(
            $endPoint,
            'POST',
            []
        );

        return [
            'response'    => $response,
            'payload'     => ['contactId' => $contactId, 'campaignId' => $campaignId],
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Update a contact using OAuth2.
     *
     * @param string $contactId
     * @param array $fieldMapData
     *
     * @return array
     */
    public function updateContact($contactId, $fieldMapData)
    {
        $endPoint = $this->baseUrl . 'contacts/' . $contactId;

        $response = $this->http->request(
            $endPoint,
            'PUT',
            JSON::encode($fieldMapData)
        );

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Add a contact to a workflow using OAuth2.
     *
     * @param string $contactId
     * @param string $workflowId
     *
     * @return array
     */
    public function addContactToWorkflow($contactId, $workflowId)
    {
        $endPoint = $this->baseUrl . 'contacts/' . $contactId . '/workflow/' . $workflowId;

        $response = $this->http->request(
            $endPoint,
            'POST',
            []
        );

        return [
            'response'    => $response,
            'payload'     => ['contactId' => $contactId, 'workflowId' => $workflowId],
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Add a task to a contact using OAuth2.
     *
     * @param string $contactId
     * @param array $fieldMapData
     *
     * @return array
     */
    public function addTask($contactId, $fieldMapData)
    {
        $endPoint = $this->baseUrl . 'contacts/' . $contactId . '/tasks';

        $response = $this->http->request(
            $endPoint,
            'POST',
            JSON::encode($fieldMapData)
        );

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Update a task using OAuth2.
     *
     * @param string $contactId
     * @param string $taskId
     * @param array $fieldMapData
     *
     * @return array
     */
    public function updateTask($contactId, $taskId, $fieldMapData)
    {
        $endPoint = $this->baseUrl . 'contacts/' . $contactId . '/tasks/' . $taskId;

        $response = $this->http->request(
            $endPoint,
            'PUT',
            JSON::encode($fieldMapData)
        );

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Add an opportunity using OAuth2.
     *
     * @param array $fieldMapData
     *
     * @return array
     */
    public function addOpportunity($fieldMapData)
    {
        $endPoint = $this->baseUrl . 'opportunities/';

        $response = $this->http->request(
            $endPoint,
            'POST',
            JSON::encode($fieldMapData)
        );

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Update an opportunity using OAuth2.
     *
     * @param string $opportunityId
     * @param array $fieldMapData
     *
     * @return array
     */
    public function updateOpportunity($opportunityId, $fieldMapData)
    {
        $endPoint = $this->baseUrl . 'opportunities/' . $opportunityId;

        $response = $this->http->request(
            $endPoint,
            'PUT',
            JSON::encode($fieldMapData)
        );

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }
}
