<?php

namespace BitApps\PiPro\src\Integrations\Twilio;

if (!\defined('ABSPATH')) {
    exit;
}

class TwilioService
{
    private const BASE_URL = 'https://api.twilio.com/2010-04-01/Accounts';

    private $http;

    private $accountSid;

    /**
     * TwilioService constructor.
     *
     * @param mixed  $httpClient
     * @param string $accountSid
     */
    public function __construct($httpClient, $accountSid)
    {
        $this->http = $httpClient;
        $this->accountSid = $accountSid;
    }

    /**
     * Send SMS Message.
     *
     * @param mixed $fieldMapData
     *
     * @return array
     */
    public function sendSms($fieldMapData)
    {
        $endPoint = self::BASE_URL . '/' . $this->accountSid . '/Messages.json';

        $smsDataArray = [
            'To'   => $fieldMapData['To'],
            'Body' => $fieldMapData['Body'],
        ];

        if (isset($fieldMapData['messageServiceId']) && !empty($fieldMapData['messageServiceId'])) {
            $smsDataArray['MessagingServiceSid'] = $fieldMapData['messageServiceId'];
        } else {
            $smsDataArray['From'] = $fieldMapData['From'];
        }

        $smsData = http_build_query($smsDataArray);
        $response = $this->http->request($endPoint, 'POST', $smsData);

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Get a Message.
     *
     * @param string $messageId
     *
     * @return array
     */
    public function getMessage($messageId)
    {
        $endPoint = self::BASE_URL . '/' . $this->accountSid . '/Messages/' . $messageId . '.json';
        $response = $this->http->request($endPoint, 'GET', []);

        return [
            'response'    => $response,
            'payload'     => $messageId,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Get a Call.
     *
     * @param string $callId
     *
     * @return array
     */
    public function getCall($callId)
    {
        $endPoint = self::BASE_URL . '/' . $this->accountSid . '/Calls/' . $callId . '.json';
        $response = $this->http->request($endPoint, 'GET', []);

        return [
            'response'    => $response,
            'payload'     => $callId,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Get all Messages.
     *
     * @param mixed $filterSwitch
     * @param mixed $fieldMapData
     *
     * @return array
     */
    public function getAllMessages($filterSwitch, $fieldMapData)
    {
        $baseEndPoint = self::BASE_URL . '/' . $this->accountSid . '/Messages.json';

        if ($filterSwitch === true) {
            $endPoint = add_query_arg(
                [
                    'to'   => $fieldMapData['To'],
                    'from' => $fieldMapData['From'],
                ],
                $baseEndPoint
            );
        } else {
            $endPoint = add_query_arg('pageSize', $fieldMapData['pageSize'], $baseEndPoint);
        }

        $response = $this->http->request($endPoint, 'GET', []);

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Get all Calls.
     *
     * @param mixed $filterSwitch
     * @param mixed $fieldMapData
     *
     * @return array
     */
    public function getAllCalls($filterSwitch, $fieldMapData)
    {
        $baseEndPoint = self::BASE_URL . '/' . $this->accountSid . '/Calls.json';

        if ($filterSwitch === true) {
            $endPoint = add_query_arg(
                [
                    'to'   => $fieldMapData['To'],
                    'from' => $fieldMapData['From'],
                ],
                $baseEndPoint
            );
        } else {
            $endPoint = add_query_arg('pageSize', $fieldMapData['pageSize'], $baseEndPoint);
        }

        $response = $this->http->request($endPoint, 'GET', []);

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Make Voice Call.
     *
     * @param mixed $fieldMapData
     *
     * @return array
     */
    public function cellPhone($fieldMapData)
    {
        $endPoint = self::BASE_URL . '/' . $this->accountSid . '/Calls.json';

        $twiml = '<?xml version="1.0" encoding="UTF-8"?>';
        $twiml .= '<Response>';
        $twiml .= '<Say voice="' . $fieldMapData['voice'] . '" language="' . $fieldMapData['language'] . '">' . htmlspecialchars($fieldMapData['message']) . '</Say>';
        $twiml .= '</Response>';

        $callData = [
            'From'  => $fieldMapData['From'],
            'To'    => $fieldMapData['To'],
            'Twiml' => $twiml
        ];

        $response = $this->http->request($endPoint, 'POST', http_build_query($callData));

        return [
            'response'    => $response,
            'payload'     => $callData,
            'status_code' => $this->http->getResponseCode()
        ];
    }
}
