<?php

namespace BitApps\PiPro\src\Integrations\Zoom;

use BitApps\Pi\Deps\BitApps\WPKit\Helpers\JSON;
use BitApps\Pi\Deps\BitApps\WPKit\Http\Client\HttpClient;

if (!\defined('ABSPATH')) {
    exit;
}
class ZoomService
{
    private const BASE_URL = 'https://api.zoom.us/v2';

    private $http;

    private $headers;

    public function __construct($headers)
    {
        $this->http = new HttpClient();
        $this->headers = $headers;
    }

    /**
     * Create a Zoom Meeting.
     *
     * @param array $fieldMapData
     *
     * @return array
     */
    public function createMeeting($fieldMapData)
    {
        $endPoint = self::BASE_URL . '/users/me/meetings';
        $response = $this->http->request($endPoint, 'POST', JSON::encode($fieldMapData), $this->headers);

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Get a Zoom Meeting by ID.
     *
     * @param int $meetingId
     *
     * @return array
     */
    public function getMeeting($meetingId)
    {
        $endPoint = self::BASE_URL . '/meetings/' . $meetingId;
        $response = $this->http->request($endPoint, 'GET', [], $this->headers);

        return [
            'response'    => $response,
            'payload'     => ['meetingId' => $meetingId],
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Get Zoom Meetings.
     *
     * @param mixed $fieldMapData
     *
     * @return array
     */
    public function getManyMeetings($fieldMapData)
    {
        $baseUrl = self::BASE_URL . '/users/me/meetings';
        $params = [];

        if (!empty($fieldMapData['returnAll'])) {
            $params['type'] = 'all';
        } else {
            if (!empty($fieldMapData['type'])) {
                $params['type'] = $fieldMapData['type'];
            }
            if (!empty($fieldMapData['limit'])) {
                $params['page_size'] = (int) $fieldMapData['limit'];
            }
        }

        $query = http_build_query($params);
        $endPoint = $baseUrl . ($query ? ('?' . $query) : '');
        $response = $this->http->request($endPoint, 'GET', [], $this->headers);

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Create a Zoom Meeting.
     *
     * @param array $fieldMapData
     *
     * @return array
     */
    public function updateMeeting($fieldMapData)
    {
        $baseUrl = self::BASE_URL . '/meetings/' . $fieldMapData['meeting'];
        $params = [];

        if (!empty($occurrenceId)) {
            $params['occurrence_id'] = $fieldMapData['occurrenceId'];
        }

        $query = http_build_query($params);
        $endPoint = $baseUrl . ($query ? ('?' . $query) : '');

        $response = $this->http->request($endPoint, 'PATCH', JSON::encode($fieldMapData), $this->headers);

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Delete Zoom by Meeting ID and Occurrence Id.
     *
     * @param int $meetingId
     * @param null|int $occurrenceId
     * @param mixed $scheduleReminder
     *
     * @return array
     */
    public function deleteMeeting($meetingId, $occurrenceId = null, $scheduleReminder = null)
    {
        $baseUrl = self::BASE_URL . '/meetings/' . $meetingId;
        $params = [];

        if (!empty($occurrenceId)) {
            $params['occurrence_id'] = $occurrenceId;
        }

        if (!empty($scheduleReminder)) {
            $params['schedule_for_reminder'] = 'true';
        }

        $query = http_build_query($params);
        $endPoint = $baseUrl . ($query ? ('?' . $query) : '');
        $response = $this->http->request($endPoint, 'DELETE', [], $this->headers);

        return [
            'response' => $response,
            'payload'  => [
                'meetingId'        => $meetingId,
                'occurrenceId'     => $occurrenceId,
                'scheduleReminder' => $scheduleReminder
            ],
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Create a Zoom Webinar.
     *
     * @param array $fieldMapData
     *
     * @return array
     */
    public function createWebinar($fieldMapData)
    {
        $endPoint = self::BASE_URL . '/users/me/webinars';
        $response = $this->http->request($endPoint, 'POST', JSON::encode($fieldMapData), $this->headers);

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }

    /**
     * Send Chat Message to Zoom.
     *
     * @param array $fieldMapData
     * @param mixed $targetType
     *
     * @return array
     */
    public function sendChatMessage($fieldMapData, $targetType)
    {
        if ($targetType === 'contact') {
            unset($fieldMapData['to_channel']);
        } elseif ($targetType === 'channel') {
            unset($fieldMapData['to_contact']);
        }

        $endPoint = self::BASE_URL . '/chat/users/me/messages';
        $response = $this->http->request($endPoint, 'POST', JSON::encode($fieldMapData), $this->headers);

        return [
            'response'    => $response,
            'payload'     => $fieldMapData,
            'status_code' => $this->http->getResponseCode()
        ];
    }
}
